//
//  WXKScheduleAppDelegate.m
//  Kyopon Schedule Sync
//
//  Created by FUJIDANA on 06/09/26.
//  Copyright 2006 FUJIDANA. All rights reserved.
//

#import "WXKScheduleAppDelegate.h"
#import <SyncServices/SyncServices.h>
#import "WXKScheduleFilter.h"


static NSString *scheduleSyncClientIdentifier                    = @"jp.sourceforge.kyoponutilities.KyoponScheduleSync";

static NSString *kAppleCalendarSchemaEventEntityName             = @"com.apple.calendars.Event";
static NSString *kAppleCalendarSchemaSummaryPropertyName         = @"summary";
static NSString *kAppleCalendarSchemaDescriptionPropertyName     = @"description";
static NSString *kAppleCalendarSchemaLocationPropertyName        = @"location";
static NSString *kAppleCalendarSchemaStartDatePropertyName       = @"start date";
static NSString *kAppleCalendarSchemaEndDatePropertyName         = @"end date";
static NSString *kAppleCalendarSchemaAllDayPropertyName          = @"all day";
static NSString *kAppleCalendarSchemaClassificationPropertyName  = @"classification";
static NSString *kAppleCalendarSchemaURLPropertyName             = @"url";

static NSString *kAppleCalendarSchemaAudioAlarmEntityName        = @"com.apple.calendars.AudioAlarm";
static NSString *kAppleCalendarSchemaTriggerdatePropertyName     = @"triggerdate";
static NSString *kAppleCalendarSchemaTriggerdurationPropertyName = @"triggerduration";
static NSString *kAppleCalendarSchemaOwnerPropertyName           = @"owner";

static NSString *WXKScheduleAppDelegateFilterDateFromDefaultKey  = @"ScheduleSyncFilterDateFrom";
static NSString *WXKScheduleAppDelegateFilterDateToDefaultKey    = @"ScheduleSyncFilterDateTo";


@interface WXKScheduleAppDelegate (Sync_Private)

- (NSManagedObject *)scheduleAddedWithChange:(ISyncChange *)change;
- (NSManagedObject *)alarmAddedWithChange:(ISyncChange *)change keyedSchedules:(NSDictionary *)keyedSchedules;

@end


#pragma mark -


@implementation WXKScheduleAppDelegate

+ (void)initialize
{
	NSDictionary *registrationDictionary = [NSDictionary dictionaryWithObjectsAndKeys:
		[NSDate dateWithTimeIntervalSinceNow:(- 30.0 * 24 * 60 * 60)], 
		WXKScheduleAppDelegateFilterDateFromDefaultKey, 
		[NSDate dateWithTimeIntervalSinceNow:(30.0 * 24 * 60 * 60)],   
		WXKScheduleAppDelegateFilterDateToDefaultKey,
		nil];
		
	[[NSUserDefaults standardUserDefaults] registerDefaults:registrationDictionary];
}

- (void)applicationWillFinishLaunching:(NSNotification *)aNotification
{
	[self setSyncController:self];
	[self setupSyncClient];
}

- (void)dealloc
{
	[_syncClient release];
	[super dealloc];
}

- (void)setupSyncClient
{
	ISyncManager *syncManager = [ISyncManager sharedManager];
	_syncClient = [[syncManager clientWithIdentifier:scheduleSyncClientIdentifier] retain];
	
	if (_syncClient == nil) {
		NSString *descriptionFile = [[NSBundle mainBundle] pathForResource:@"ScheduleSyncDescription" ofType:@"plist"];
		_syncClient = [[syncManager registerClientWithIdentifier:scheduleSyncClientIdentifier descriptionFilePath:descriptionFile] retain];
	}
	
	// Currently always uses "pull the truth"
	[_syncClient setShouldReplaceClientRecords:YES forEntityNames:[_syncClient supportedEntityNames]];
}

- (ISyncClient *)syncClient
{
	if (_syncClient == nil) {
		[self setupSyncClient];
	}
	return _syncClient;
}

- (NSArray *)syncFilters
{
	NSUserDefaults *userDefaults = [NSUserDefaults standardUserDefaults];
	
	NSDate *fromDate = [userDefaults objectForKey:WXKScheduleAppDelegateFilterDateFromDefaultKey];
	NSDate *toDate   = [userDefaults objectForKey:WXKScheduleAppDelegateFilterDateToDefaultKey];
	WXKScheduleFilter *scheduleFilter = [[WXKScheduleFilter alloc] initWithRangeOfDateFrom:fromDate to:toDate];
	NSArray *result = [NSArray arrayWithObject:scheduleFilter];
	[scheduleFilter release];
	return result;
}

// This method is invoked when a session return shouldReplaceAllRecordsOnClientForEntityName:.
- (void)removeAllLocalRecordsOfEntityName:(NSString *)entityName
{
	NSManagedObjectContext *context      = [self managedObjectContext];
	NSFetchRequest         *fetchRequest = [[[NSFetchRequest alloc] init] autorelease];
	NSEntityDescription    *entity       = nil;
	
	if ([entityName isEqualToString:kAppleCalendarSchemaEventEntityName]) {
		entity = [NSEntityDescription entityForName:@"Schedule" inManagedObjectContext:context];
	}
	
	[context processPendingChanges];
	
	if (entity != nil) {
		[fetchRequest setEntity:entity];
		NSArray *results = [context executeFetchRequest:fetchRequest error:NULL];
		if (results) {
			NSEnumerator *resultsEnumerator = [results objectEnumerator];
			NSManagedObject *object;
			while (object = [resultsEnumerator nextObject]) {
				[context deleteObject:object];
			}
		} 
	}
	[context processPendingChanges];
}

- (void)acceptOrDenyChangesWithEntityNames:(NSArray *)entityNames session:(ISyncSession *)session
{
	NSEnumerator *changeEnumerator;
	ISyncChange  *change;
	
	NSMutableDictionary *keyedSchedules = [NSMutableDictionary dictionary];
	
	if ([entityNames containsObject:kAppleCalendarSchemaEventEntityName]) {
		changeEnumerator = [session changeEnumeratorForEntityNames:[NSArray arrayWithObject:kAppleCalendarSchemaEventEntityName]];
		
		while (change = [changeEnumerator nextObject]) {
			switch ([change type]) {
				case ISyncChangeTypeAdd:
				{
					id schedule = [self scheduleAddedWithChange:change];
					[keyedSchedules setObject:schedule forKey:[change recordIdentifier]];
					break;
				}
				case ISyncChangeTypeModify:
					NSLog(@"Modified: %@", [change record]);
					break;
				case ISyncChangeTypeDelete:
					NSLog(@"deleted: %@", [change record]);
					break;
			}
		}
	}
	
	if ([entityNames containsObject:kAppleCalendarSchemaAudioAlarmEntityName]) {
		changeEnumerator = [session changeEnumeratorForEntityNames:[NSArray arrayWithObject:kAppleCalendarSchemaAudioAlarmEntityName]];
		
		while (change = [changeEnumerator nextObject]) {
			switch ([change type]) {
				case ISyncChangeTypeAdd:
				{
					[self alarmAddedWithChange:change keyedSchedules:keyedSchedules];
					break;
				}
				case ISyncChangeTypeModify:
					NSLog(@"Modified: %@", [change record]);
					break;
				case ISyncChangeTypeDelete:
					NSLog(@"deleted: %@", [change record]);
					break;
			}
		}
	}
}


@end


#pragma mark -


@implementation WXKScheduleAppDelegate (Sync_Private)

- (NSManagedObject *)scheduleAddedWithChange:(ISyncChange *)change
{
	NSDictionary *changeRecord = [change record];
	
	// if entity of added record is com.apple.calendars.Event...
	if ([[changeRecord objectForKey:ISyncRecordEntityNameKey] isEqualToString:kAppleCalendarSchemaEventEntityName]) {
		NSManagedObject *schedule = [NSEntityDescription insertNewObjectForEntityForName:@"Schedule" inManagedObjectContext:[self managedObjectContext]];
		[schedule setValue:[change recordIdentifier] forKey:@"recordIdentifier"];
		
		id value;
		if (value = [changeRecord objectForKey:kAppleCalendarSchemaSummaryPropertyName]) {
			[schedule setValue:value forKey:@"summary"];
		}
		if (value = [changeRecord objectForKey:kAppleCalendarSchemaDescriptionPropertyName]) {
			[schedule setValue:value forKey:@"details"];
		}
		if (value = [changeRecord objectForKey:kAppleCalendarSchemaLocationPropertyName]) {
			[schedule setValue:value forKey:@"location"];
		}
		if (value = [changeRecord objectForKey:kAppleCalendarSchemaStartDatePropertyName]) {
			[schedule setValue:value forKey:@"startDate"];
		}
		if (value = [changeRecord objectForKey:kAppleCalendarSchemaEndDatePropertyName]) {
			[schedule setValue:value forKey:@"endDate"];
		}
		if (value = [changeRecord objectForKey:kAppleCalendarSchemaAllDayPropertyName]) {
			[schedule setValue:value forKey:@"allDay"];
		}
		if (value = [changeRecord objectForKey:kAppleCalendarSchemaClassificationPropertyName]) {
			[schedule setValue:value forKey:@"classification"];
		}
		if (value = [changeRecord objectForKey:kAppleCalendarSchemaURLPropertyName]) {
			[schedule setValue:[value absoluteString] forKey:@"urlString"];
		}
		return schedule;
	}
	return nil;
}

- (NSManagedObject *)alarmAddedWithChange:(ISyncChange *)change keyedSchedules:(NSDictionary *)keyedSchedules
{
	NSDictionary *changeRecord = [change record];
	NSLog(@"%@", changeRecord);
	
	// check wheter the entity of added record is com.apple.calendars.AudioAlarm...
	if ([[changeRecord objectForKey:ISyncRecordEntityNameKey] isEqualToString:kAppleCalendarSchemaAudioAlarmEntityName] == NO) {
		return nil;
	}
	
	id value = nil;
	id owner = nil;
	if (value = [[changeRecord objectForKey:kAppleCalendarSchemaOwnerPropertyName] lastObject]) {
		owner = [keyedSchedules objectForKey:value];
	}
	if (owner == nil) {
		return nil;
	}
	
	NSDate *startDate = [owner valueForKey:@"startDate"];
	if (startDate == nil) {
		return nil;
	}
	
	NSDate *triggerDate;
	if (value = [changeRecord objectForKey:kAppleCalendarSchemaTriggerdatePropertyName]) {
		triggerDate = value;
	} else if (value = [changeRecord objectForKey:kAppleCalendarSchemaTriggerdurationPropertyName]) {
		triggerDate = [startDate addTimeInterval:[value intValue]];
	} else {
		return nil;
	}
	
	NSManagedObject *alarm = [NSEntityDescription insertNewObjectForEntityForName:@"Alarm" inManagedObjectContext:[self managedObjectContext]];
	[alarm setValue:[change recordIdentifier] forKey:@"recordIdentifier"];
	[alarm setValue:owner forKey:@"owner"];
	[alarm setValue:triggerDate forKey:@"triggerDate"];
	
	return alarm;
}

@end

